using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using EQUIPMENT = gov.va.med.vbecs.Common.VbecsTables.Equipment;
using ADDRESS = gov.va.med.vbecs.Common.VbecsTables.Address;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Krzysztof Dobranowski</Developer>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/15/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access component implementing database read/write functionality
	///	for equipment-related data
	///</summary>

	public class Equipment
	{
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="99"> 
		///		<ExpectedInput>Equipment Type, Manufacturer Name, Serial Number</ExpectedInput>
		///		<ExpectedOutput>DataTable of matching Equipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="100"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Searches for in-service equipment satisfying the specified search criteria
		/// </summary>
		/// <param name="equipmentType">Equipment type</param>
		/// <param name="manufacturerName">Manufacturer name</param>
		/// <param name="serialNumber">Serial number</param>
		/// <returns>DataTable containing a list of equipment found</returns>
		public static DataTable GetEquipment(	string equipmentType, 
												string manufacturerName, 
												string serialNumber)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipment.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetEquipment.equipmenttype, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetEquipment.manufacturername, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetEquipment.serialnumber, System.Data.SqlDbType.VarChar)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			
			if ((equipmentType== null)||(equipmentType.Trim().Length == 0))
			{
				prms[1].Value = null;
			}
			else
			{
				prms[1].Value = equipmentType;
			}
			
			if ((manufacturerName == null) || (manufacturerName.Trim().Length == 0))
			{
				prms[2].Value = null;
			}
			else
			{
				prms[2].Value = manufacturerName;
			}
			
			if ((serialNumber == null) || (serialNumber.Trim().Length == 0))
			{
				prms[3].Value = null;
			}
			else
			{
				prms[3].Value = serialNumber;
			}
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetEquipment.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5654"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of all equipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5655"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets all equipment items (including out of service) for a division
		/// </summary>
		/// <returns>Equipment data table</returns>
		public static DataTable GetEquipment()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipment.divisioncode, System.Data.SqlDbType.Char),
				new SqlParameter(STOREDPROC.GetEquipment.equipmenttype, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetEquipment.manufacturername, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetEquipment.serialnumber, System.Data.SqlDbType.VarChar),
				new SqlParameter("@includeall", System.Data.SqlDbType.Bit)
			};

			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = null;
			prms[2].Value = null;
			prms[3].Value = null;
			prms[4].Value = true;
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetEquipment.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1181"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable of Equipment for division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1182"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get previously entered equipment types and manufacturers as a data set
		/// </summary>
		/// <returns>Equipment type and Manufacturer name data tables</returns>
		public static DataSet GetEquipmentDetails()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipmentDetails.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentDetails.StoredProcName, prms);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="101"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataTable schema of Equipment Table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="102"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a blank Equipment data table
		/// </summary>
		/// <returns>Equipment data table</returns>
		public static System.Data.DataTable GetEmptyEquipmentTable()
		{
			System.Data.DataTable dtEmpty = new System.Data.DataTable(EQUIPMENT.TableName);

			dtEmpty.Columns.Add(EQUIPMENT.Comments, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.EquipmentGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(EQUIPMENT.EquipmentName, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.InspectionIndicator, typeof(bool));
			dtEmpty.Columns.Add(EQUIPMENT.OutOfServiceIndicator, typeof(bool));
			dtEmpty.Columns.Add(EQUIPMENT.EquipmentType, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.InvoiceNumber, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.LocalIdentifier, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.MaintenanceContractDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(EQUIPMENT.MaintenanceContractNumber, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.ManufacturerName, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.ManufacturerPhoneNumber, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.NextMaintenanceText, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.ReceiptDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(EQUIPMENT.ReceiptTechId, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.RowVersion, typeof(byte[]));
			dtEmpty.Columns.Add(EQUIPMENT.SerialNumber, typeof(string));
			dtEmpty.Columns.Add(EQUIPMENT.AddressGuid, typeof(Guid));

			return(dtEmpty);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="103"> 
		///		<ExpectedInput>EquipmentGuid</ExpectedInput>
		///		<ExpectedOutput>DataTable of Equipment Details</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="104"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets details of an Equipment item
		/// </summary>
		/// <param name="equipmentGuid">Equipment unique identifier</param>
		/// <returns>Equipment data table</returns>
		public static System.Data.DataTable GetEquipmentByGuid(System.Guid equipmentGuid)
		{
			SqlParameter[] _params = 
			{
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetEquipmentByGuid.equipmentguid, equipmentGuid),
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetEquipmentByGuid.divisioncode, Common.LogonUser.LogonUserDivisionCode)
			};

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentByGuid.StoredProcName, _params).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5712"> 
		///		<ExpectedInput>Valid serial number and manufacturer name string</ExpectedInput>
		///		<ExpectedOutput>Data table containing equipment guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5713"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns equipment guids of any existing equipment matching serial number and manufacturer name
		/// </summary>
		/// <param name="serialNumber">Serial Number</param>
		/// <param name="manufacturerName">Manufacturer Name</param>
		/// <returns>Data table of equipment guids</returns>
		public static System.Data.DataTable GetEquipmentBySerialNumberAndManufacturer(string serialNumber, string manufacturerName)
		{
			SqlParameter[] _params = 
			{
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetEquipmentBySerialNumberAndManufacturer.serialnumber, serialNumber),
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetEquipmentBySerialNumberAndManufacturer.manufacturername, manufacturerName),
				Common.StoredProcedure.MakeParameter(STOREDPROC.GetEquipmentBySerialNumberAndManufacturer.divisioncode, Common.LogonUser.LogonUserDivisionCode)
			};

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentBySerialNumberAndManufacturer.StoredProcName, _params).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5656"> 
		///		<ExpectedInput>Valid equipment and address tables</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5657"> 
		///		<ExpectedInput>Invalid equipment and address tables containing duplicate entries</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException thrown </ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Insert equipment into the database
		/// </summary>
		/// <param name="dtEquipment">Data table of Equipment to be inserted</param>
		/// <param name="dtAddress">Data table of associated manufacturer addresses to be inserted</param>
		/// <param name="lastUpdateFunctionId">Last update function</param>
		/// <param name="dtWorkloadEvents">Workload Events</param>
		/// <returns>Success indicator</returns>
		public static bool Insert(DataTable dtEquipment, DataTable dtAddress, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable[] dtArray = new DataTable[3];
			string[] spArray = new string[3];
			//
			dtArray[0] = Common.Utility.AppendLastUpdateInformation(dtAddress,lastUpdateFunctionId);
			spArray[0] = STOREDPROC.InsertAddress.StoredProcName;
			//
			dtArray[1] = Common.Utility.AppendLastUpdateInformation(dtEquipment,lastUpdateFunctionId);
			spArray[1] = STOREDPROC.InsertEquipment.StoredProcName;

			dtArray[2] = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
			spArray[2] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
			
			return (new StoredProcedure().TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5658"> 
		///		<ExpectedInput>Valid equipment, address and maintenance event tables</ExpectedInput>
		///		<ExpectedOutput>Success indicator = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5659"> 
		///		<ExpectedInput>Invalid equipment and address tables (row version of null)</ExpectedInput>
		///		<ExpectedOutput>ArgumentException thrown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates equipment and equipment maintenance events in the database
		/// </summary>
		/// <param name="dtEquipment">Data table of Equipment to be updated</param>
		/// <param name="dtAddress">Data table of associated manufacturer addresses to be updated</param>
		/// <param name="insertMaintenanceEvent">Data table of maintenance events to be inserted</param>
		/// <param name="updateMaintenanceEvent">Data table of maintenance events to be updated</param>
		/// <param name="lastUpdateFunctionId">Last update function</param>
		/// <param name="dtWorkloadEvents"></param>
		/// <returns>Success indicator</returns>
		public static bool Update(DataTable dtEquipment, DataTable dtAddress, DataTable insertMaintenanceEvent, DataTable updateMaintenanceEvent, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
			DataTable[] dtArray = new DataTable[5];
			string[] spArray = new string[5];
			//
			dtArray[0] = Common.Utility.AppendLastUpdateInformation(dtAddress,lastUpdateFunctionId);
			spArray[0] = STOREDPROC.UpdateAddress.StoredProcName;
			//
			dtArray[1] = Common.Utility.AppendLastUpdateInformation(dtEquipment,lastUpdateFunctionId);
			spArray[1] = STOREDPROC.UpdateEquipment.StoredProcName;
			//
			dtArray[2] = Common.Utility.AppendLastUpdateInformation(insertMaintenanceEvent,lastUpdateFunctionId);
			spArray[2] = STOREDPROC.InsertEquipmentMaintenance.StoredProcName;
			//
			dtArray[3] = Common.Utility.AppendLastUpdateInformation(updateMaintenanceEvent,lastUpdateFunctionId);
			spArray[3] = STOREDPROC.UpdateEquipmentMaintenance.StoredProcName;

			if(dtWorkloadEvents != null)
			{
				dtArray[4] = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
				spArray[4] = STOREDPROC.InsertWorkloadEvents.StoredProcName;
			}
			
			return (new StoredProcedure().TransactionalGetValue(spArray, dtArray) == 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5660"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of equipment types</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5661"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get equipment types entered previously
		/// </summary>
		/// <returns>Data table of equipment types</returns>
		public static DataTable GetEquipmentTypes()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipmentTypes.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentTypes.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5662"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Data table of equipment manufacturer names</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5663"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get manufacturer names entered previously
		/// </summary>
		/// <returns>Data table of manufacturer names</returns>
		public static DataTable GetEquipmentManufacturerNames()
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipmentManufacturers.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentManufacturers.StoredProcName, prms).Tables[0];
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5664"> 
		///		<ExpectedInput>Valid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Data table of manufacturer address and phone numbers</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5665"> 
		///		<ExpectedInput>Unknown manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get addresses and phone numbers entered previously for a manufacturer
		/// </summary>
		/// <param name="manufacturername">Manufacturer name</param>
		/// <returns>Data table of address and phone number info</returns>
		public static DataTable GetManufacturerAddresses(string manufacturername)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetManufacturerAddresses.manufacturername, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = manufacturername;

			return Common.StoredProcedure.GetData(STOREDPROC.GetManufacturerAddresses.StoredProcName, prms).Tables[0];

		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5666"> 
		///		<ExpectedInput>Valid equipment guid</ExpectedInput>
		///		<ExpectedOutput>Equipment report data set</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5667"> 
		///		<ExpectedInput>Invalid equipment guid</ExpectedInput>
		///		<ExpectedOutput>0 record count</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get equipment report data set
		/// </summary>
		/// <param name="equipmentGuid">Equipment unique identifier</param>
		/// <returns>Equipment and equipment maintenance data set</returns>
		public static DataSet GetEquipmentReport(Guid equipmentGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetEquipmentRecordForReport.equipmentguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetEquipmentRecordForReport.divisioncode, System.Data.SqlDbType.Char)
			};
			
			prms[0].Value = equipmentGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return Common.StoredProcedure.GetData(STOREDPROC.GetEquipmentRecordForReport.StoredProcName, prms);
		}
	}
}
